package com.logicbig.example;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import static org.junit.jupiter.api.Assertions.*;

public class NestedClassStateInheritanceTest {
    // Shared state initialized in outer class
    protected UserRepository userRepository;
    protected LocalDateTime testStartTime;
    protected List<String> auditLog;

    @BeforeEach
    void setUpSharedState() {
        // Common setup that ALL nested tests will inherit
        testStartTime = LocalDateTime.now();
        auditLog = new ArrayList<>();
        userRepository = new UserRepository();

        // Initialize with common test data
        userRepository.addUser(new User("admin", "admin@example.com", User.Role.ADMIN));
        userRepository.addUser(new User("user1", "user1@example.com", User.Role.USER));

        auditLog.add("Outer setup completed at " + testStartTime);
        System.out.println("✓ Outer @BeforeEach: Shared state initialized");
    }

    @AfterEach
    void tearDownSharedState() {
        auditLog.add("Test completed at " + LocalDateTime.now());
        System.out.println("Audit Log entries: " + auditLog.size());
    }

    @Test
    void outerTest_sharedStateAvailable() {
        // Outer tests can use the shared state
        assertNotNull(userRepository);
        assertNotNull(testStartTime);
        assertEquals(2, userRepository.count());
        System.out.println("Outer test accessing shared repository");
    }

    @Nested
    class UserAuthenticationTests {
        private String authToken;

        @BeforeEach
        void setUpAuthentication() {
            // Additional setup specific to authentication tests
            // Outer class state is already available!
            authToken = "token-" + System.currentTimeMillis();
            auditLog.add("Auth setup: token generated");
            System.out.println("✓ Auth @BeforeEach: Using shared userRepository with "
                    + userRepository.count() + " users");
        }

        @Test
        void authenticateAdminUser() {
            // Accessing outer class state
            User admin = userRepository.findByUsername("admin");
            assertNotNull(admin);
            assertEquals(User.Role.ADMIN, admin.getRole());

            // Using nested class specific state
            assertNotNull(authToken);
            assertTrue(authToken.startsWith("token-"));

            auditLog.add("Admin authenticated with token");
            System.out.println("Auth test using shared repository and local authToken");
        }

        @Test
        void authenticateRegularUser() {
            User user = userRepository.findByUsername("user1");
            assertNotNull(user);
            assertEquals(User.Role.USER, user.getRole());

            auditLog.add("User authenticated");
        }

        @Nested
        class PasswordValidationTests {

            @BeforeEach
            void setUpPasswordValidation() {
                // Can access both outer and parent nested class state
                auditLog.add("Password validation setup");
                System.out.println("✓ Password @BeforeEach: Has access to authToken? " +
                        (authToken != null));
            }

            @Test
            void validatePasswordStrength() {
                // Accessing state from all ancestor levels
                assertNotNull(userRepository);  // From outer class
                assertNotNull(authToken);       // From parent nested class
                assertNotNull(testStartTime);   // From outer class

                User user = userRepository.findByUsername("user1");
                assertTrue(user.validatePassword("StrongPass123!"));

                auditLog.add("Password validated");
                System.out.println("Password test accessing state from all ancestor levels");
            }
        }
    }

    @Nested
    class UserManagementTests {

        @BeforeEach
        void setUpUserManagement() {
            // Different nested class, same shared outer state
            // Adding users specific to management tests
            userRepository.addUser(new User("manager",
                    "manager@example.com", User.Role.MANAGER));
            auditLog.add("Manager user added for management tests");
            System.out.println("✓ Management @BeforeEach: Repository now has "
                    + userRepository.count() + " users");
        }

        @Test
        void createNewUser() {
            int initialCount = userRepository.count();
            User newUser = new User("newuser", "new@example.com", User.Role.USER);
            userRepository.addUser(newUser);

            assertEquals(initialCount + 1, userRepository.count());
            assertNotNull(userRepository.findByUsername("newuser"));

            auditLog.add("New user created");
        }

        @Test
        void deleteUser() {
            assertTrue(userRepository.deleteUser("user1"));
            assertNull(userRepository.findByUsername("user1"));

            auditLog.add("User deleted");
        }

        @Nested
        class BulkOperationsTests {

            @Test
            void bulkUserImport() {
                // Accessing and modifying shared state
                int originalCount = userRepository.count();

                // Simulate bulk import
                for (int i = 1; i <= 5; i++) {
                    User user = new User("bulk" + i, "bulk" + i +
                            "@example.com", User.Role.USER);
                    userRepository.addUser(user);
                }

                assertEquals(originalCount + 5, userRepository.count());
                auditLog.add("Bulk import completed: added 5 users");
                System.out.println("Bulk test modified shared repository");
            }
        }
    }

    @Nested
    class AuditLogTests {

        @Test
        void verifyAuditLogIntegration() {
            // Demonstrates how all tests contribute to shared audit log
            assertNotNull(auditLog);
            System.out.println("Audit log entries from all tests: " + auditLog);

            // The audit log contains entries from all nested test setups
            assertTrue(auditLog.size() > 0);
        }
    }
}

// Supporting classes for the example
class User {
    enum Role {ADMIN, USER, MANAGER}

    private final String username;
    private final String email;
    private final Role role;

    public User(String username, String email, Role role) {
        this.username = username;
        this.email = email;
        this.role = role;
    }

    public String getUsername() {return username;}

    public String getEmail() {return email;}

    public Role getRole() {return role;}

    public boolean validatePassword(String password) {
        return password != null && password.length() >= 8;
    }
}

class UserRepository {
    private final Map<String, User> users = new HashMap<>();

    public void addUser(User user) {
        users.put(user.getUsername(), user);
    }

    public User findByUsername(String username) {
        return users.get(username);
    }

    public boolean deleteUser(String username) {
        return users.remove(username) != null;
    }

    public int count() {
        return users.size();
    }
}