package com.logicbig.example;

import org.junit.jupiter.api.MediaType;
import org.junit.jupiter.api.extension.AfterTestExecutionCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.ExtensionContext.Namespace;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;

public class TestReportingExtension implements BeforeEachCallback, AfterTestExecutionCallback {

    private static final DateTimeFormatter TIMESTAMP_FORMATTER =
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS");

    private static final String START_TIME_KEY = "testStartTime";
    private static final Namespace NAMESPACE =
            Namespace.create(TestReportingExtension.class, "timing");

    @Override
    public void beforeEach(ExtensionContext context) {
        long startTime = System.currentTimeMillis();
        context.getStore(NAMESPACE).put(START_TIME_KEY, startTime);

        // Publish test start information
        context.publishReportEntry("Test Start", LocalDateTime.now().format(TIMESTAMP_FORMATTER));

        Map<String, String> testMetadata = new HashMap<>();
        testMetadata.put("testClass", context.getRequiredTestClass().getSimpleName());
        testMetadata.put("testMethod", context.getRequiredTestMethod().getName());
        testMetadata.put("displayName", context.getDisplayName());

        context.publishReportEntry(testMetadata);
    }

    @Override
    public void afterTestExecution(ExtensionContext context) {


        // Determine test outcome
        String status = context.getExecutionException().isPresent() ? "FAILED" : "PASSED";

        // 1. Publish key-value report entries
        context.publishReportEntry("status", status);
        long testDuration;
        Long startTime = context.getStore(NAMESPACE).get(START_TIME_KEY, Long.class);
        if (startTime != null) {
            testDuration = System.currentTimeMillis() - startTime;
            context.publishReportEntry("duration_ms", String.valueOf(testDuration));
        } else {testDuration = -1;}

        // Clean up
        context.getStore(NAMESPACE).remove(START_TIME_KEY);


        // 2. Publish value-only entry for simple status
        context.publishReportEntry("Test execution " + status.toLowerCase());

        // 3. Publish multiple entries at once
        Map<String, String> executionDetails = new HashMap<>();
        executionDetails.put("timestamp", LocalDateTime.now().format(TIMESTAMP_FORMATTER));
        if (testDuration != -1) {
            executionDetails.put("executionTime", testDuration + " ms");
        }
        executionDetails.put("thread", Thread.currentThread().getName());

        context.getExecutionException().ifPresent(exception -> {
            executionDetails.put("exception", exception.getClass().getSimpleName());
            executionDetails.put("exceptionMessage", exception.getMessage());
        });

        context.publishReportEntry(executionDetails);

        // 4. Publish a detailed log file
        String logFileName = context.getRequiredTestMethod().getName() + "-report.log";
        context.publishFile(
                logFileName,
                MediaType.TEXT_PLAIN,
                path -> Files.writeString(path, "test logs content")
        );

        // 5. Publish an artifact directory with test outputs
        String artifactDirName = context.getRequiredTestMethod().getName() + "-artifacts";
        context.publishDirectory(
                artifactDirName,
                dir -> createArtifacts(dir, context, testDuration, status)
        );
    }

    private void createArtifacts(Path directory,
                                 ExtensionContext context,
                                 long duration,
                                 String status)
            throws IOException {
        Files.createDirectories(directory);

        // Create summary file
        Path summaryFile = directory.resolve("summary.txt");
        Files.writeString(summaryFile,
                          "Test: " + context.getDisplayName() + "\n" +
                                  "Status: " + status + "\n" +
                                  "Generated: " + LocalDateTime.now().format(TIMESTAMP_FORMATTER) + "\n" +
                                  "Artifact count: 3"
        );


        System.out.println("[EXTENSION] Artifact directory created: " + directory.getFileName());
        System.out.println("[EXTENSION] Files created: summary.txt");
    }
}