package com.logicbig.example;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public enum PhoneFormat {
    STANDARD("(\\d{3})-(\\d{3})-(\\d{4})") {
        @Override
        String format(Phone phone) {
            return phone.getAreaCode() +
                    "-" + phone.getExchangeCode() +
                    "-" + phone.getLineNumber();
        }
    },
    PLAIN("(\\d{3})(\\d{3})(\\d{4})") {
        @Override
        String format(Phone phone) {
            return phone.getAreaCode() +
                    phone.getExchangeCode() +
                    phone.getLineNumber();
        }
    };

    private final Pattern compiledPattern;

    PhoneFormat(String pattern) {
        compiledPattern = Pattern.compile(pattern);
    }

    public static PhoneFormat getPhoneFormat(String formatString) {
        try {
            return valueOf(formatString.toUpperCase());
        } catch (IllegalArgumentException | NullPointerException e) {
            //returning default format;
            return STANDARD;
        }
    }

    public static Phone parse(String formatString, String value) {
        PhoneFormat phoneFormat = getPhoneFormat(formatString);
        Matcher matcher = phoneFormat.compiledPattern.matcher(value);
        if (!matcher.matches()) {
            throw new RuntimeException(
                    String.format("invalid input: %s. The valid format is of type %s%n",
                            value, phoneFormat.compiledPattern.pattern()));
        }
        String areaCode = matcher.group(1);
        String exchange = matcher.group(2);
        String line = matcher.group(3);
        return new Phone(areaCode, exchange, line);
    }

    public static String format(String formatString, Phone phone) {
        PhoneFormat phoneFormat = getPhoneFormat(formatString);
        return phoneFormat.format(phone);
    }

    abstract String format(Phone phone);
}