package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import java.util.List;

public class ExampleMain {
    static EntityManagerFactory emf = Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {

        try {
            persistEntity();
            usePredicate();
            useNegatePredicate();
            useExpressionPredicate();
            useBooleanExpression();
            useNegateBooleanExpression();
        } finally {
            emf.close();
        }
    }

    private static void useBooleanExpression() {
        System.out.println("-- finding person who is registered voter --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Person> query = criteriaBuilder.createQuery(Person.class);
        Root<Person> person = query.from(Person.class);
        //in following line person.get() returns Path which is sub-interface of Expression
        Expression<Boolean> voterExpression = person.get(Person_.registeredVoter);
        query.select(person)
             .where(voterExpression);
        TypedQuery<Person> typedQuery = entityManager.createQuery(query);
        List<Person> p = typedQuery.getResultList();
        p.forEach(System.out::println);
    }

    private static void useNegateBooleanExpression() {
        System.out.println("-- finding person who is not registered voter --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Person> query = criteriaBuilder.createQuery(Person.class);
        Root<Person> person = query.from(Person.class);
        //in following line person.get() returns Path which is sub-interface of Expression
        Expression<Boolean> voterExpression = person.get(Person_.registeredVoter);
        query.select(person)
             .where(criteriaBuilder.isFalse(voterExpression));
        TypedQuery<Person> typedQuery = entityManager.createQuery(query);
        List<Person> p = typedQuery.getResultList();
        p.forEach(System.out::println);
    }

    private static void useExpressionPredicate() {
        System.out.println("-- finding person with null age --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Person> query = criteriaBuilder.createQuery(Person.class);
        Root<Person> person = query.from(Person.class);
        //in following line person.get() returns Path which is sub-interface of Expression
        Expression<Byte> ageExpression = person.get(Person_.age);
        Predicate nullAgePredicate = ageExpression.isNull();
        query.select(person)
             .where(nullAgePredicate);
        TypedQuery<Person> typedQuery = entityManager.createQuery(query);
        Person p = typedQuery.getSingleResult();
        System.out.println(p);
    }

    private static void usePredicate() {
        System.out.println("-- finding person named 'Lindsey' --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Person> query = criteriaBuilder.createQuery(Person.class);
        Root<Person> person = query.from(Person.class);
        Predicate predicate = criteriaBuilder.equal(person.get(Person_.name), "Lindsey");
        query.select(person)
             .where(predicate);
        TypedQuery<Person> typedQuery = entityManager.createQuery(query);
        Person p = typedQuery.getSingleResult();
        System.out.println(p);
    }

    private static void useNegatePredicate() {
        System.out.println("-- finding person not named 'Lindsey' --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Person> query = criteriaBuilder.createQuery(Person.class);
        Root<Person> person = query.from(Person.class);
        Predicate predicate = criteriaBuilder.equal(person.get(Person_.name), "Lindsey");
        query.select(person)
             .where(predicate.not());
        TypedQuery<Person> typedQuery = entityManager.createQuery(query);
        List<Person> p = typedQuery.getResultList();
        p.forEach(System.out::println);
    }

    private static void persistEntity() {
        System.out.println("-- Persisting entities --");
        EntityManager em = emf.createEntityManager();

        Person p1 = Person.of("Lindsey", (byte) 31, false);
        Person p2 = Person.of("Morgan", (byte) 40, true);
        Person p3 = Person.of("Mike", null, false);

        em.getTransaction().begin();
        em.persist(p1);
        em.persist(p2);
        em.persist(p3);
        em.getTransaction().commit();
        em.close();
    }
}