package com.logicbig.example;

import javax.persistence.*;
import javax.persistence.criteria.*;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            findEmployeeFullName();
            findLongestNamedEmployee();
            findAllCities();
            findByCity();
            findByDept();
        } finally {
            entityManagerFactory.close();
        }
    }

    private static void findEmployeeFullName() {
        System.out.println("-- Employees full names --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<String> query = criteriaBuilder.createQuery(String.class);//create query object
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.select(
                criteriaBuilder.concat(
                        criteriaBuilder.concat(employeeRoot.get(Employee_.FIRST_NAME), " "),
                        employeeRoot.get(Employee_.LAST_NAME)));
        TypedQuery<String> typedQuery = entityManager.createQuery(query);
        typedQuery.getResultList()
                  .forEach(System.out::println);
        entityManager.close();
    }

    private static void findLongestNamedEmployee() {
        System.out.println("-- Find employee having the longest full name --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);//create query object
        Root<Employee> employeeRoot = query.from(Employee.class);

        Subquery<Integer> maxLengthSubQuery = query.subquery(Integer.class);
        Root<Employee> subQueryEmployeeRoot = maxLengthSubQuery.from(Employee.class);
        maxLengthSubQuery.select(
                criteriaBuilder.max(criteriaBuilder.length(
                        criteriaBuilder.concat(
                                criteriaBuilder.concat(subQueryEmployeeRoot.get(Employee_.FIRST_NAME), " "),
                                subQueryEmployeeRoot.get(Employee_.LAST_NAME))))
        );

        query.select(employeeRoot)
             .where(criteriaBuilder.equal(
                     criteriaBuilder.length(
                             criteriaBuilder.concat(
                                     criteriaBuilder.concat(employeeRoot.get(Employee_.FIRST_NAME), " "),
                                     employeeRoot.get(Employee_.LAST_NAME))
                     ),
                     maxLengthSubQuery)
             );

        TypedQuery<Employee> typedQuery = entityManager.createQuery(query);
        typedQuery.getResultList()
                  .forEach(System.out::println);
        entityManager.close();
    }


    private static void findAllCities() {
        System.out.println("-- All Cities --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<String> query = criteriaBuilder.createQuery(String.class);//create query object
        Root<Employee> employeeRoot = query.from(Employee.class);

        Expression<String> citySubstringExpr = createCitySubstringExpression(criteriaBuilder, employeeRoot);
        query.select(citySubstringExpr)
             .distinct(true);

        List<String> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(System.out::println);
        entityManager.close();
    }

    private static Expression<String> createCitySubstringExpression(CriteriaBuilder criteriaBuilder,
                                                                    Root<Employee> employeeRoot) {
        return criteriaBuilder.trim(
                criteriaBuilder.substring(employeeRoot.get(Employee_.ADDRESS),
                        criteriaBuilder.sum(
                                criteriaBuilder.locate(employeeRoot.get(Employee_.ADDRESS), ","),
                                1))
        );
    }

    private static void findByCity() {
        System.out.println("-- Find By City 'Sunland' --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.select(employeeRoot)
             .where(criteriaBuilder.equal(createCitySubstringExpression(criteriaBuilder, employeeRoot), "Sunland"));
        entityManager.createQuery(query)
                     .getResultList()
                     .forEach(System.out::println);
        entityManager.close();
    }

    private static void findByDept() {
        System.out.println("-- Find By Dept 'Admin' ignore case --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.select(employeeRoot)
             .where(criteriaBuilder.equal(
                     criteriaBuilder.lower(employeeRoot.get(Employee_.dept))
                     , "admin"));
        entityManager.createQuery(query)
                     .getResultList()
                     .forEach(System.out::println);
    }

    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", "John", "IT", "111 Round Drive, Sunland");
        Employee employee2 = Employee.create("Rose", "Burgess", "Admin", "2623  Elmwood Avenue, Scottsdale");
        Employee employee3 = Employee.create("Denise", "Clarkson", "admin", "4000 Johnny Lane, Mars Town");
        Employee employee4 = Employee.create("Mike", "Young", "IT", "323 Tik Tok Road, Hiland");
        Employee employee5 = Employee.create("Linda", "McDonald", "Sales", "2000 Wet Road, Sunland");
        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.persist(employee5);
        em.getTransaction().commit();

        System.out.println("-- Employees persisted --");
        Query query = em.createQuery(
                "SELECT e FROM Employee e");
        List<Employee> resultList = query.getResultList();
        resultList.forEach(System.out::println);
        em.close();
    }
}