package com.logicbig.example;

import javax.persistence.*;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Root;
import javax.persistence.criteria.Subquery;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            //show some pre info
            showNetAvgSalary();
            findEmployeeWithLessThanAverageSalary();
            showAvgSalaryByDept();
            findDeptHavingAboveNetAverageSalary();
        } finally {
            entityManagerFactory.close();
        }
    }

    private static void showNetAvgSalary() {
        System.out.println("-- net average salary --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Double> netAvgSalaryQuery = criteriaBuilder.createQuery(Double.class);
        Root<Employee> employee = netAvgSalaryQuery.from(Employee.class);
        netAvgSalaryQuery.select(criteriaBuilder.avg(employee.get(Employee_.salary)));
        Double netAvgSalary = entityManager.createQuery(netAvgSalaryQuery).getSingleResult();
        System.out.println(netAvgSalary);
    }

    private static void showAvgSalaryByDept() {
        System.out.println("-- Dept by average salaries --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Tuple> netAvgSalaryQueryByDept = criteriaBuilder.createTupleQuery();
        Root<Employee> employee = netAvgSalaryQueryByDept.from(Employee.class);
        netAvgSalaryQueryByDept.groupBy(employee.get(Employee_.dept));
        netAvgSalaryQueryByDept.multiselect(employee.get(Employee_.dept),
                criteriaBuilder.avg(employee.get(Employee_.salary)));
        TypedQuery<Tuple> typedQuery = entityManager.createQuery(netAvgSalaryQueryByDept);
        List<Tuple> resultList = typedQuery.getResultList();
        resultList.forEach(tuple ->
                System.out.printf("Dept: %s, Avg Salary: %s%n", tuple.get(0), tuple.get(1)));
    }

    private static void findEmployeeWithLessThanAverageSalary() {
        System.out.println("-- Employees who have less than average salary --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        //main query
        CriteriaQuery<Employee> employeeQuery = criteriaBuilder.createQuery(Employee.class);
        //main query from
        Root<Employee> employee = employeeQuery.from(Employee.class);

        //create subquery
        Subquery<Double> averageSalarySubQuery = employeeQuery.subquery(Double.class);
        //subquery from
        Root<Employee> subQueryEmployee = averageSalarySubQuery.from(Employee.class);
        //subquery selection
        averageSalarySubQuery.select(criteriaBuilder.avg(subQueryEmployee.get(Employee_.salary)));

        //main query selection
        employeeQuery.select(employee)
                     .where(criteriaBuilder
                             .lessThan(employee.get(Employee_.salary).as(Double.class), averageSalarySubQuery));
        TypedQuery<Employee> typedQuery = entityManager.createQuery(employeeQuery);
        List<Employee> employees = typedQuery.getResultList();
        employees.forEach(System.out::println);
        entityManager.close();
    }

    private static void findDeptHavingAboveNetAverageSalary() {
        System.out.println("-- Dept having AVG salaries greater than net AVG salary --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        //main query
        CriteriaQuery<Tuple> netAvgSalaryQueryByDept = criteriaBuilder.createTupleQuery();
        Root<Employee> employee = netAvgSalaryQueryByDept.from(Employee.class);
        netAvgSalaryQueryByDept.groupBy(employee.get(Employee_.dept));
        netAvgSalaryQueryByDept.multiselect(employee.get(Employee_.dept),
                criteriaBuilder.avg(employee.get(Employee_.salary)));
        //subquery
        Subquery<Double> netAvgSalarySubquery = netAvgSalaryQueryByDept.subquery(Double.class);
        Root<Employee> subQueryEmployee = netAvgSalarySubquery.from(Employee.class);
        netAvgSalarySubquery.select(criteriaBuilder.avg(subQueryEmployee.get(Employee_.salary)));
        //main query with having restriction
        netAvgSalaryQueryByDept.having(
                criteriaBuilder.greaterThan(criteriaBuilder.avg(employee.get(Employee_.salary)), netAvgSalarySubquery)
        );
        TypedQuery<Tuple> typedQuery = entityManager.createQuery(netAvgSalaryQueryByDept);
        List<Tuple> resultList = typedQuery.getResultList();
        resultList.forEach(tuple ->
                System.out.printf("Dept: %s, Avg Salary: %s%n", tuple.get(0), tuple.get(1)));

    }


    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", "IT", 3000);
        Employee employee2 = Employee.create("Rose", "Admin", 2000);
        Employee employee3 = Employee.create("Denise", "Admin", 4000);
        Employee employee4 = Employee.create("Mike", "IT", 3500);
        Employee employee5 = Employee.create("Linda", "Sales", 2000);
        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.persist(employee5);
        em.getTransaction().commit();
        em.close();
        System.out.println("-- all employees --");
        System.out.println(employee1);
        System.out.println(employee2);
        System.out.println(employee3);
        System.out.println(employee4);
        System.out.println(employee5);
    }
}