package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Root;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            findEmployeeCountGroupByDept();
            findEmployeeAvgSalariesGroupByDept();
            findEmployeeCountGroupBySalary();
            findEmployeeMaxSalariesGroupBySelectedDept();
        } finally {
            entityManagerFactory.close();
        }
    }

    private static void findEmployeeCountGroupByDept() {
        System.out.println("-- Employee count group by dept --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.groupBy(employee.get(Employee_.dept));
        query.multiselect(employee.get(Employee_.dept), criteriaBuilder.count(employee));
        TypedQuery<Object[]> typedQuery = entityManager.createQuery(query);
        List<Object[]> resultList = typedQuery.getResultList();
        resultList.forEach(objects ->
                System.out.printf("Dept: %s, Count: %s%n", objects[0], objects[1]));
        entityManager.close();
    }

    private static void findEmployeeAvgSalariesGroupByDept() {
        System.out.println("-- Employee avg salaries group by dept --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.groupBy(employee.get(Employee_.dept));
        query.multiselect(employee.get(Employee_.dept),
                criteriaBuilder.avg(employee.get(Employee_.salary)));
        TypedQuery<Object[]> typedQuery = entityManager.createQuery(query);
        List<Object[]> resultList = typedQuery.getResultList();
        resultList.forEach(objects ->
                System.out.printf("Dept: %s, Avg Salary: %s%n", objects[0], objects[1]));
        entityManager.close();
    }

    private static void findEmployeeCountGroupBySalary() {
        System.out.println("-- Employee count group by salary --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.groupBy(employee.get(Employee_.salary));
        query.multiselect(employee.get(Employee_.salary), criteriaBuilder.count(employee));
        TypedQuery<Object[]> typedQuery = entityManager.createQuery(query);
        List<Object[]> resultList = typedQuery.getResultList();
        resultList.forEach(objects ->
                System.out.printf("Salary: %s, Count: %s%n", objects[0], objects[1]));
        entityManager.close();
    }


    private static void findEmployeeMaxSalariesGroupBySelectedDept() {
        System.out.println("-- Employees max salary group by dept - only in IT and Admin dept --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.groupBy(employee.get(Employee_.dept));
        query.multiselect(employee.get(Employee_.dept),
                criteriaBuilder.max(employee.get(Employee_.salary)));
        query.having(employee.get(Employee_.dept).in("IT", "Admin"));
        TypedQuery<Object[]> typedQuery = entityManager.createQuery(query);
        List<Object[]> resultList = typedQuery.getResultList();
        resultList.forEach(objects ->
                System.out.printf("Dept: %s, Max Salary: %s%n", objects[0], objects[1]));
        entityManager.close();
    }

    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", "IT", 3000);
        Employee employee2 = Employee.create("Rose", "Admin", 2000);
        Employee employee3 = Employee.create("Denise", "Admin", 4000);
        Employee employee4 = Employee.create("Mike", "IT", 3500);
        Employee employee5 = Employee.create("Linda", "Sales", 2000);
        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.persist(employee5);
        em.getTransaction().commit();
        em.close();
        System.out.println("-- all employees --");
        System.out.println(employee1);
        System.out.println(employee2);
        System.out.println(employee3);
        System.out.println(employee4);
        System.out.println(employee5);
    }
}