package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import java.util.ArrayList;
import java.util.List;

public class ExampleMain {

    public static void main(String[] args) throws Exception {
        EntityManagerFactory emf =
                Persistence.createEntityManagerFactory("example-unit");
        try {
            persistEntities(emf);
            getByPartTimeEmployeesWeeklySalary(emf);
            getByPartTimeAndContractEmployeesBySalary(emf);
            getProjectByFullTimeEmployeesSalary(emf);
            getProjectBySupervisorPartTimeSalary(emf);
        } finally {
            emf.close();
        }
    }

    //using method: Root<T> treat(Root<X> root, Class<T> type);
    private static void getByPartTimeEmployeesWeeklySalary(EntityManagerFactory emf) {
        System.out.println("-- employees where PartTimeEmployee.weeklySalary < 1000 --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employee = query.from(Employee.class);
        Root<PartTimeEmployee> partTimeEmployee = criteriaBuilder.treat(employee, PartTimeEmployee.class);
        query.select(partTimeEmployee)
             .distinct(true)
             .where(criteriaBuilder.lessThan(partTimeEmployee.get(PartTimeEmployee_.weeklySalary), 1000));
        TypedQuery<Employee> typedQuery = entityManager.createQuery(query);
        List<Employee> employees = typedQuery.getResultList();
        employees.forEach(System.out::println);
        entityManager.close();
    }

    //using method: Root<T> treat(Root<X> root, Class<T> type);
    private static void getByPartTimeAndContractEmployeesBySalary(EntityManagerFactory emf) {
        System.out.println("-- employees where PartTimeEmployee.weeklySalary > 1000 or "
                + "ContractEmployee#hourlyRate < 75 --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employee = query.from(Employee.class);
        Root<PartTimeEmployee> partTimeEmployee = criteriaBuilder.treat(employee, PartTimeEmployee.class);
        Root<ContractEmployee> contractEmployee = criteriaBuilder.treat(employee, ContractEmployee.class);
        query.select(partTimeEmployee)
             .distinct(true)
             .where(criteriaBuilder.or(
                     criteriaBuilder.greaterThan(partTimeEmployee.get(PartTimeEmployee_.weeklySalary), 1000),
                     criteriaBuilder.lessThan(contractEmployee.get(ContractEmployee_.hourlyRate), 75)
             ));
        TypedQuery<Employee> typedQuery = entityManager.createQuery(query);
        List<Employee> employees = typedQuery.getResultList();
        employees.forEach(System.out::println);
        entityManager.close();
    }

    //using method: ListJoin<X, E> treat(ListJoin<X, T> join, Class<E> type)
    private static void getProjectByFullTimeEmployeesSalary(EntityManagerFactory emf) {
        System.out.println("-- getting projects where any FullTimeEmployee.annualSalary > 100000 by using join --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Project> query = criteriaBuilder.createQuery(Project.class);
        Root<Project> project = query.from(Project.class);
        ListJoin<Project, Employee> employees = project.join(Project_.employees);
        ListJoin<Project, FullTimeEmployee> fullTimeEmployee = criteriaBuilder.treat(employees, FullTimeEmployee.class);
        query.select(project)
             .distinct(true)
             .where(criteriaBuilder.greaterThan(fullTimeEmployee.get(FullTimeEmployee_.annualSalary), 100000));
        TypedQuery<Project> typedQuery = entityManager.createQuery(query);
        List<Project> projects = typedQuery.getResultList();
        projects.forEach(System.out::println);
        entityManager.close();
    }

    //using method:  Path<T> treat(Path<X> path, Class<T> type);
    private static void getProjectBySupervisorPartTimeSalary(EntityManagerFactory emf) {
        System.out.println("-- Projects where ((PartTimeEmployee)Project.supervisor).weeklySalary > 1000 --");
        EntityManager entityManager = emf.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Project> query = criteriaBuilder.createQuery(Project.class);
        Root<Project> project = query.from(Project.class);
        Path<Employee> supervisor = project.get(Project_.supervisor);
        Path<PartTimeEmployee> partTimeSupervisor = criteriaBuilder.treat(supervisor, PartTimeEmployee.class);
        query.select(project)
             .where(criteriaBuilder.greaterThan(partTimeSupervisor.get(PartTimeEmployee_.weeklySalary), 1000));
        TypedQuery<Project> typedQuery = entityManager.createQuery(query);
        List<Project> projects = typedQuery.getResultList();
        projects.forEach(System.out::println);
        entityManager.close();
    }

    private static void persistEntities(EntityManagerFactory emf) throws Exception {
        System.out.println("-- Persisting entities --");
        EntityManager em = emf.createEntityManager();
        em.getTransaction().begin();
        for (Project project : createProjects()) {
            em.persist(project);
            System.out.println(project);
        }
        em.getTransaction().commit();
    }

    private static List<Project> createProjects() {
        List<Project> projects = new ArrayList<>();

        FullTimeEmployee e1 = new FullTimeEmployee();
        e1.setName("Sara");
        e1.setAnnualSalary(120000);

        PartTimeEmployee e2 = new PartTimeEmployee();
        e2.setName("Jon");
        e2.setWeeklySalary(900);

        ContractEmployee e3 = new ContractEmployee();
        e3.setName("Tom");
        e3.setHourlyRate(60);

        PartTimeEmployee supervisor1 = new PartTimeEmployee();
        supervisor1.setName("Tina");
        supervisor1.setWeeklySalary(1300);

        projects.add(Project.create(supervisor1, "Trade UI", e1, e2, e3));

        FullTimeEmployee e4 = new FullTimeEmployee();
        e4.setName("Mike");
        e4.setAnnualSalary(80000);

        PartTimeEmployee e5 = new PartTimeEmployee();
        e5.setName("Jackie");
        e5.setWeeklySalary(1200);

        ContractEmployee e6 = new ContractEmployee();
        e6.setName("Aly");
        e6.setHourlyRate(90);

        PartTimeEmployee supervisor2 = new PartTimeEmployee();
        supervisor2.setName("Trish");
        supervisor2.setWeeklySalary(900);

        projects.add(Project.create(supervisor2, "Broker UI", e4, e5, e6));

        return projects;
    }
}