package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.Query;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.ListJoin;
import javax.persistence.criteria.Root;
import java.util.Arrays;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            findNumberOfPhoneNumbers();
            findEmployeeWithMoreThanOnePhoneNumber();
            findEmployeeWithMoreThanOnePhoneNumber2();
            findPhoneByIndex();

        } finally {
            entityManagerFactory.close();
        }
    }

    private static void findNumberOfPhoneNumbers() {
        System.out.println("-- Find number of phone numbers --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.multiselect(employeeRoot.get(Employee_.id), employeeRoot.get(Employee_.name),
                criteriaBuilder.size(employeeRoot.get(Employee_.phoneNumbers.getName())))
             .groupBy(employeeRoot.get(Employee_.id), employeeRoot.get(Employee_.name));
        //The equivalent JPQL:
        //SELECT e.id, e.name, SIZE(e.phoneNumbers) from Employee e GROUP BY e.id, e.name

        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(o -> System.out.println(Arrays.toString(o)));
        entityManager.close();
    }

    private static void findEmployeeWithMoreThanOnePhoneNumber() {
        System.out.println("-- Find employees who have more than 1 phone number by using size() --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.select(employeeRoot)
             .where(criteriaBuilder.greaterThan(criteriaBuilder.size(employeeRoot.get(Employee_.phoneNumbers)), 1));
        //The equivalent JPQL:
        //SELECT e from Employee e  WHERE SIZE(e.phoneNumbers) > 1

        entityManager.createQuery(query)
                     .getResultList()
                     .forEach(System.out::println);
    }

    private static void findEmployeeWithMoreThanOnePhoneNumber2() {
        System.out.println("-- Find employees who have more than 1 phone number by using index() --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        ListJoin<Object, Object> phoneNumbersJoin = employeeRoot.joinList(Employee_.phoneNumbers.getName());
        query.select(employeeRoot)
             .distinct(true)
             .where(criteriaBuilder.greaterThan(phoneNumbersJoin.index(), 0));
        //The equivalent JPQL:
        //SELECT DISTINCT e from Employee e JOIN e.phoneNumbers p WHERE INDEX(p) > 0

        entityManager.createQuery(query)
                     .getResultList()
                     .forEach(System.out::println);
    }

    private static void findPhoneByIndex() {
        System.out.println("-- Find phones by index in phoneNumbers list --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        ListJoin<Employee, String> phoneNumbersJoin = employeeRoot.joinList(Employee_.phoneNumbers.getName());

        query.multiselect(employeeRoot.get(Employee_.name), phoneNumbersJoin.index(), phoneNumbersJoin);
        //The equivalent JPQL:
        //Select e.name, INDEX(p), p from Employee e JOIN e.phoneNumbers p

        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(o -> System.out.println(Arrays.toString(o)));
        entityManager.close();
    }

    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", 2000, "IT", "111-111-111", "666-666-666", "777-777-777");
        Employee employee2 = Employee.create("Rose", 3500, "Admin", "222-222,222", "888-888-888");
        Employee employee3 = Employee.create("Denise", 2500, "Admin", "333-333-333");
        Employee employee4 = Employee.create("Mike", 4000, "IT", "444-444-444");
        Employee employee5 = Employee.create("Linda", 4500, "Sales", "555-555-555");
        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.persist(employee5);
        em.getTransaction().commit();

        System.out.println("-- Employees persisted --");
        Query query = em.createQuery(
                "SELECT e FROM Employee e");
        List<Employee> resultList = query.getResultList();
        resultList.forEach(System.out::println);
        em.close();
    }
}