package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Root;
import java.util.Arrays;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            findAllEmployeesViaSimpleCase();
            findAllEmployeesViaGeneralCase();
            findAllEmployeesSalaryViaGeneralCase();
        } finally {
            entityManagerFactory.close();
        }
    }

    private static void findAllEmployeesViaSimpleCase() {
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        System.out.println("-- Finding all employees with simple case --");
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.multiselect(employee.get(Employee_.name),
                employee.get(Employee_.salary),
                criteriaBuilder.selectCase(employee.get(Employee_.DEPT))
                               .when("IT", "Information Technology")
                               .when("Admin", "Administration")
                               .otherwise(employee.get(Employee_.DEPT))
        );
        //The equivalent JPQL
         /* SELECT e.name, e.salary,
                         CASE e.dept
                         WHEN 'IT' THEN 'Information Technology'
                         WHEN 'Admin' THEN 'Administration'
                         ELSE e.dept
                         END
                         FROM Employee e*/
        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(arr -> System.out.println(Arrays.toString(arr)));
        entityManager.close();
    }

    private static void findAllEmployeesViaGeneralCase() {
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        System.out.println("-- Finding all employees with general case --");
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.multiselect(employee.get(Employee_.name),
                employee.get(Employee_.salary),
                criteriaBuilder.selectCase()
                               .when(criteriaBuilder
                                       .equal(employee.get(Employee_.DEPT), "IT"), "Information Technology")
                               .when(criteriaBuilder.equal(employee.get(Employee_.DEPT), "Admin"), "Administration")
                               .otherwise(employee.get(Employee_.DEPT))
        );
        //The equivalent JPQL
         /* SELECT e.name, e.salary,
                         CASE
                         WHEN e.dept='IT' THEN 'Information Technology'
                         WHEN e.dept='Admin' THEN 'Administration'
                         ELSE e.dept
                         END
                         FROM Employee e*/
        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(arr -> System.out.println(Arrays.toString(arr)));
        entityManager.close();
    }

    private static void findAllEmployeesSalaryViaGeneralCase() {
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        System.out.println("-- Finding all employees salary with general case --");
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employee = query.from(Employee.class);
        query.multiselect(employee.get(Employee_.name),
                employee.get(Employee_.salary),
                criteriaBuilder.selectCase()
                               .when(criteriaBuilder.lessThan(employee.get(Employee_.SALARY), 3000),
                                       "Salary Less than 3000")
                               .when(criteriaBuilder.between(employee.get(Employee_.SALARY), 3000, 4000),
                                       "Salary between 3000 and 4000")
                               .otherwise("Salary more than 4000")
        );

        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(arr -> System.out.println(Arrays.toString(arr)));
        entityManager.close();
    }


    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", 2000, "IT");
        Employee employee2 = Employee.create("Rose", 3000, "Admin");
        Employee employee3 = Employee.create("Denise", 2500, "Admin");
        Employee employee4 = Employee.create("Mike", 4000, "IT");
        Employee employee5 = Employee.create("Linda", 4500, "Sales");
        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.persist(employee5);
        em.getTransaction().commit();
        em.close();
        System.out.println("-- employee persisted --");
        System.out.println(employee1);
        System.out.println(employee2);
        System.out.println(employee3);
        System.out.println(employee4);
        System.out.println(employee5);
    }
}