package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Join;
import javax.persistence.criteria.Root;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            findEmployeeBySalaryRange();
            findEmployeeBySalaryRange2();
            findEmployeeByJoinDate();
        } finally {
            entityManagerFactory.close();
        }
    }

    private static void findEmployeeBySalaryRange() {
        System.out.println("-- Employee with salary BETWEEN 2000 and 4000 --");
        EntityManager em = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = em.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employee = query.from(Employee.class);
        query.select(employee)
             .where(criteriaBuilder.between(employee.get(Employee_.salary), 2000L, 4000L));
        List<Employee> resultList = em.createQuery(query).getResultList();
        resultList.forEach(System.out::println);
        em.close();
    }

    private static void findEmployeeBySalaryRange2() {
        System.out.println("-- Employee with salary BETWEEN supervisor's salary and 4000 --");
        EntityManager em = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = em.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employee = query.from(Employee.class);
        Join<Employee, Employee> supervisor = employee.join(Employee_.supervisor);
        query.select(employee)
             .where(criteriaBuilder.between(employee.get(Employee_.salary),
                     supervisor.get(Employee_.salary), criteriaBuilder.literal(4000L)));
        List<Employee> resultList = em.createQuery(query).getResultList();
        resultList.forEach(System.out::println);
        em.close();
    }

    private static void findEmployeeByJoinDate() {
        System.out.println("-- Employee with join date BETWEEN '1990-01-01' and '2010-05-01' --");
        EntityManager em = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = em.getCriteriaBuilder();
        CriteriaQuery<Employee> query = criteriaBuilder.createQuery(Employee.class);
        Root<Employee> employee = query.from(Employee.class);
        query.select(employee)
             .where(criteriaBuilder.between(employee.get(Employee_.joinDate),
                     localToTimeStamp(LocalDate.of(1990, 01, 01)),
                     localToTimeStamp(LocalDate.of(2010, 05, 01))))
             .orderBy(criteriaBuilder.desc(employee.get(Employee_.joinDate)));
        List<Employee> resultList = em.createQuery(query).getResultList();
        resultList.forEach(System.out::println);
        em.close();
    }

    private static Timestamp localToTimeStamp(LocalDate date) {
        return Timestamp.from(date.atStartOfDay().toInstant(ZoneOffset.UTC));
    }

    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", 3000, LocalDate.of(1999, 11, 15));
        Employee employee2 = Employee.create("Rose", 4000, LocalDate.of(2011, 5, 1));
        Employee employee3 = Employee.create("Denise", 1500, LocalDate.of(2006, 1, 10));
        Employee employee4 = Employee.create("Mike", 2000, LocalDate.of(2015, 8, 20));
        employee1.setSupervisor(employee4);
        employee2.setSupervisor(employee1);
        employee3.setSupervisor(employee1);
        employee4.setSupervisor(null);


        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.getTransaction().commit();
        em.close();
        System.out.println("-- employee persisted --");
        System.out.println(employee1);
        System.out.println(employee2);
        System.out.println(employee3);
        System.out.println(employee4);
    }
}