package com.logicbig.example;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.Query;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Root;
import java.util.Arrays;
import java.util.List;

public class ExampleMain {
    private static EntityManagerFactory entityManagerFactory =
            Persistence.createEntityManagerFactory("example-unit");

    public static void main(String[] args) {
        try {
            persistEmployees();
            findAbsoluteDifferenceWithAvgSalary();
            findStandardDeviationOfSalaries();
            findSalariesNearestToThousands();
            findSalariesNearestToThousands2();
        } finally {
            entityManagerFactory.close();
        }
    }

    private static void findAbsoluteDifferenceWithAvgSalary() {
        System.out.println("-- Find salaries' positive difference with average salary --");
        //first find average salary
        Double avgSal = getAverageSalary();
        System.out.println("Average Salary: " + avgSal);

        //now find positive difference with average salary
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.multiselect(employeeRoot.get(Employee_.NAME),
                criteriaBuilder.abs(
                        criteriaBuilder.diff(employeeRoot.get(Employee_.salary), avgSal)
                )
        );
        //The equivalent JPQL:
        //"SELECT e.name, ABS(e.salary - avgSal) FROM Employee e")

        entityManager.createQuery(query).getResultList()
                     .forEach(arr -> System.out.println(Arrays.toString(arr)));
        entityManager.close();
    }

    private static Double getAverageSalary() {
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Double> query = criteriaBuilder.createQuery(Double.class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.select(criteriaBuilder.avg(employeeRoot.get(Employee_.SALARY)));
        Double avgSal = entityManager.createQuery(query)
                                     .getSingleResult();
        entityManager.close();
        return avgSal;
    }

    private static void findStandardDeviationOfSalaries() {
        System.out.println("-- Find salaries' standard deviation --");
        //find average salary
        Double avgSal = getAverageSalary();

        //find square root of squared sum salary difference from avg
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();

        CriteriaQuery<Double> queryDeviationSum = criteriaBuilder.createQuery(Double.class);
        Root<Employee> employeeRoot = queryDeviationSum.from(Employee.class);
        Expression<Double> salDiffWithAvgSal = criteriaBuilder.diff(employeeRoot.get(Employee_.salary), avgSal);
        queryDeviationSum.select(criteriaBuilder.sqrt(
                criteriaBuilder.sum(criteriaBuilder.prod(salDiffWithAvgSal, salDiffWithAvgSal))));
        //The equivalent JPQL:
        //SELECT SQRT(SUM((e.salary - :avgSal)*(e.salary - :avgSal))) from Employee e
        Double deviationSum = entityManager.createQuery(queryDeviationSum).getSingleResult();

        //find square root of employee's count
        CriteriaQuery<Double> querySampleSizeSquaredRoot = criteriaBuilder.createQuery(Double.class);
        Root<Employee> employeeRoot2 = querySampleSizeSquaredRoot.from(Employee.class);
        querySampleSizeSquaredRoot.select(criteriaBuilder.sqrt(criteriaBuilder.count(employeeRoot2)));
        Double sampleSizeSquaredRoot = entityManager.createQuery(querySampleSizeSquaredRoot).getSingleResult();
        //The equivalent JPQL:
        //SELECT SQRT(COUNT(e)) from Employee e

        Double standardDeviation = deviationSum / sampleSizeSquaredRoot;
        System.out.println("standard deviation: " + standardDeviation);
        entityManager.close();
    }

    private static void findSalariesNearestToThousands() {
        System.out.println("-- Find salaries nearest to 1000 using mod() --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.multiselect(employeeRoot.get(Employee_.NAME),
                criteriaBuilder.diff(
                        employeeRoot.get(Employee_.SALARY),
                        criteriaBuilder.mod(employeeRoot.get(Employee_.SALARY), 1000)
                )
        );
        //The equivalent JPQL:
        //SELECT e.name, e.salary - MOD(e.salary, 1000) from Employee e

        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(o -> System.out.println(Arrays.toString(o)));
        entityManager.close();
    }

    private static void findSalariesNearestToThousands2() {
        System.out.println("-- Find salaries nearest to 1000 using quot() --");
        EntityManager entityManager = entityManagerFactory.createEntityManager();
        CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
        CriteriaQuery<Object[]> query = criteriaBuilder.createQuery(Object[].class);
        Root<Employee> employeeRoot = query.from(Employee.class);
        query.multiselect(employeeRoot.get(Employee_.NAME),
                criteriaBuilder.prod(
                        criteriaBuilder.quot(
                                employeeRoot.get(Employee_.SALARY).as(Integer.class),
                                1000
                        ).as(Integer.class),
                        1000
                )
        );
        //The equivalent JPQL:
        //SELECT e.name, e.salary - MOD(e.salary, 1000) from Employee e

        List<Object[]> resultList = entityManager.createQuery(query).getResultList();
        resultList.forEach(o -> System.out.println(Arrays.toString(o)));
        entityManager.close();
    }

    public static void persistEmployees() {
        Employee employee1 = Employee.create("Diana", 2000, "IT");
        Employee employee2 = Employee.create("Rose", 3500, "Admin");
        Employee employee3 = Employee.create("Denise", 2500, "Admin");
        Employee employee4 = Employee.create("Mike", 4000, "IT");
        Employee employee5 = Employee.create("Linda", 4500, "Sales");
        EntityManager em = entityManagerFactory.createEntityManager();
        em.getTransaction().begin();
        em.persist(employee1);
        em.persist(employee2);
        em.persist(employee3);
        em.persist(employee4);
        em.persist(employee5);
        em.getTransaction().commit();

        System.out.println("-- Employees persisted --");
        Query query = em.createQuery(
                "SELECT e FROM Employee e");
        List<Employee> resultList = query.getResultList();
        resultList.forEach(System.out::println);
        em.close();
    }
}