package com.logicbig.example;

import javax.validation.*;
import javax.validation.constraints.NotNull;
import java.util.Set;

public class ConstraintPayloadExample2 {
    private static final Validator validator;

    static {
        Configuration<?> config = Validation.byDefaultProvider().configure();
        ValidatorFactory factory = config.buildValidatorFactory();
        validator = factory.getValidator();
        factory.close();
    }

    public interface AppErrorHandler<T> extends Payload {
        void onError (ConstraintViolation<T> violation);
    }

    public static class ErrorEmailSender<T> implements AppErrorHandler<T> {
        @Override
        public void onError (ConstraintViolation<T> violation) {
            System.out.println("Sending email to support team: " +
                                violation.getPropertyPath() + " " +
                                violation.getMessage());
        }
    }

    public static class TestBean {

        @NotNull(payload = {ErrorEmailSender.class})
        private String str;

        public String getStr () {
            return str;
        }

        public void setStr (String str) {
            this.str = str;
        }
    }

    public static void main (String[] args) {
        TestBean bean = new TestBean();

        Set<ConstraintViolation<TestBean>> constraintViolations =
                            validator.validate(bean);

        if (constraintViolations.size() > 0) {
            constraintViolations.stream().forEach(
                                ConstraintPayloadExample2::processError);
        } else {
            //proceed using user object
            System.out.println(bean);
        }
    }


    private static void processError (ConstraintViolation<TestBean> violation) {
        Set<Class<? extends Payload>> payload =
                            violation.getConstraintDescriptor().getPayload();

        payload.forEach(p -> {
            if (AppErrorHandler.class.isAssignableFrom(p)) {
                try {
                    AppErrorHandler errorHandler = (AppErrorHandler) p.newInstance();
                    errorHandler.onError(violation);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }
}