package com.logicbig.example;

import com.fasterxml.jackson.databind.ObjectMapper;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.model.chat.ChatModel;
import dev.langchain4j.model.chat.request.ChatRequest;
import dev.langchain4j.model.chat.request.ResponseFormat;
import dev.langchain4j.model.chat.request.ResponseFormatType;
import dev.langchain4j.model.chat.request.json.*;
import dev.langchain4j.model.chat.response.ChatResponse;
import dev.langchain4j.model.ollama.OllamaChatModel;

public class SchemaTypesExample {
    public static void main(String[] args) {
        ChatModel chatModel = OllamaChatModel.builder()
                                             .baseUrl("http://localhost:11434")
                                             .modelName("phi3:mini-128k")
                                             .logRequests(true)
                                             .logResponses(true)
                                             .build();

        // Example 1: String schema with description
        JsonSchemaElement stringSchema = JsonStringSchema.builder()
            .description("The full name of the person")
            .build();
        
        // Example 2: Integer schema
        JsonSchemaElement integerSchema = JsonIntegerSchema.builder()
            .description("Age in years")
            .build();
        
        // Example 3: Number schema for decimal values
        JsonSchemaElement numberSchema = JsonNumberSchema.builder()
            .description("Height in meters")
            .build();
        
        // Example 4: Boolean schema
        JsonSchemaElement booleanSchema = JsonBooleanSchema.builder()
            .description("Marital status")
            .build();
        
        // Example 5: Enum schema
        JsonSchemaElement enumSchema = JsonEnumSchema.builder()
            .description("Employment status")
            .enumValues(EmploymentStatus.toStringList())
            .build();
        
        // Example 6: Complete object schema
        JsonSchemaElement personSchema = JsonObjectSchema.builder()
            .addProperty("name", stringSchema)
            .addProperty("age", integerSchema)
            .addProperty("height", numberSchema)
            .addProperty("married", booleanSchema)
            .addProperty("employmentStatus", enumSchema)
            .required("name", "age")
            .build();
        
        // Create JSON Schema
        JsonSchema jsonSchema = JsonSchema.builder()
            .name("DetailedPerson")
            .rootElement(personSchema)
            .build();
        
        // Create response format
        ResponseFormat responseFormat = ResponseFormat.builder()
            .type(ResponseFormatType.JSON)
            .jsonSchema(jsonSchema)
            .build();

        String text = """
                John is 42 years old and lives an independent life. 
                He stands 1.75 meters tall. He has a his own business and works from home. 
                Currently unmarried he enjoys the freedom to focus on his personal goals.
                """;

        // Create chat request
        ChatRequest chatRequest =
                ChatRequest.builder()
                           .responseFormat(responseFormat)
                           .messages(UserMessage.from(text))
                           .build();


        // Send request and get response
        ChatResponse chatResponse = chatModel.chat(chatRequest);

        // Extract and parse JSON response
        String jsonResponse = chatResponse.aiMessage().text();
        System.out.println("JSON Response: " + jsonResponse);

        // Parse JSON to Person object
        ObjectMapper mapper = new ObjectMapper();
        try {
            Person person = mapper.readValue(jsonResponse, Person.class);
            System.out.println(person);
        } catch (Exception e) {
            System.err.println("Error parsing JSON: " + e.getMessage());
        }
        

    }
}