package com.logicbig.example;

import com.fasterxml.jackson.databind.ObjectMapper;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.model.chat.ChatModel;
import dev.langchain4j.model.chat.request.ChatRequest;
import dev.langchain4j.model.chat.request.ResponseFormat;
import dev.langchain4j.model.chat.request.ResponseFormatType;
import dev.langchain4j.model.chat.request.json.JsonObjectSchema;
import dev.langchain4j.model.chat.request.json.JsonSchema;
import dev.langchain4j.model.chat.response.ChatResponse;
import dev.langchain4j.model.ollama.OllamaChatModel;

public class PersonExtractionExample {
    public static void main(String[] args) {

        ChatModel chatModel = OllamaChatModel.builder()
                                             .baseUrl("http://localhost:11434")
                                             .modelName("phi3:mini-128k")
                                             .logRequests(true)
                                             .logResponses(true)
                                             .build();

        // Define JSON Schema for Person
        JsonSchema jsonSchema =
                JsonSchema.builder()
                          .name("Person")
                          .rootElement(JsonObjectSchema
                                               .builder()
                                               .addStringProperty("name")
                                               .addIntegerProperty("age")
                                               .addNumberProperty("height")
                                               .addBooleanProperty("married")
                                               .addEnumProperty("employmentStatus",
                                                                EmploymentStatus.toStringList())
                                               .required("name",
                                                         "age",
                                                         "height",
                                                         "married",
                                                         "employmentStatus")
                                               .build())
                          .build();

        // Create response format with JSON Schema
        ResponseFormat responseFormat = ResponseFormat.builder()
                                                      .type(ResponseFormatType.JSON)
                                                      .jsonSchema(jsonSchema)
                                                      .build();

        // Create user message with text to analyze
        String text = """
                John is 42 years old. 
                   He stands 1.75 meters tall and carries himself with confidence. 
                   He is employed in a local company as a Software Engineer. 
                   He's married with 2 kids.
                """;

        // Create chat request
        ChatRequest chatRequest =
                ChatRequest.builder()
                           .responseFormat(responseFormat)
                           .messages(UserMessage.from(text))
                           .build();


        // Send request and get response
        ChatResponse chatResponse = chatModel.chat(chatRequest);

        // Extract and parse JSON response
        String jsonResponse = chatResponse.aiMessage().text();
        System.out.println("JSON Response: " + jsonResponse);

        // Parse JSON to Person object
        ObjectMapper mapper = new ObjectMapper();
        try {
            Person person = mapper.readValue(jsonResponse, Person.class);
            System.out.println(person);
        } catch (Exception e) {
            System.err.println("Error parsing JSON: " + e.getMessage());
        }
    }
}