package com.logicbig.example;

import com.fasterxml.jackson.databind.ObjectMapper;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.model.chat.ChatModel;
import dev.langchain4j.model.chat.request.ChatRequest;
import dev.langchain4j.model.chat.request.ResponseFormat;
import dev.langchain4j.model.chat.request.ResponseFormatType;
import dev.langchain4j.model.chat.request.json.*;
import dev.langchain4j.model.chat.response.ChatResponse;
import dev.langchain4j.model.ollama.OllamaChatModel;

public class CompanyExample {
    public static void main(String[] args) {
        // Create Ollama ChatModel with phi3
        ChatModel chatModel =
                OllamaChatModel.builder()
                               .baseUrl("http://localhost:11434")
                               .modelName("phi3:mini-128k")
                               .logRequests(true)
                               .logResponses(true)
                               .build();

        // Create enum schema for roles
        JsonSchemaElement roleSchema = JsonEnumSchema.builder()
                                                     .enumValues("DEVELOPER", "DESIGNER", "MANAGER", "ANALYST")
                                                     .description("Employee role")
                                                     .build();

        // Create schema for employee
        JsonSchemaElement employeeSchema = JsonObjectSchema.builder()
                                                           .addStringProperty("name")
                                                           .addStringProperty("email")
                                                           .addProperty("role", roleSchema)
                                                           .required("name", "email", "role")
                                                           .build();

        // Create array schema for employees
        JsonSchemaElement employeesArraySchema = JsonArraySchema.builder()
                                                                .items(employeeSchema)
                                                                .description("List of company employees")
                                                                .build();

        // Create company schema with employees array
        JsonSchemaElement companySchema = JsonObjectSchema.builder()
                                                          .addStringProperty("name", "Company name")
                                                          .addProperty("employees", employeesArraySchema)
                                                          .required("name", "employees")
                                                          .build();

        // Create JSON Schema
        JsonSchema jsonSchema = JsonSchema.builder()
                                          .name("Company")
                                          .rootElement(companySchema)
                                          .build();

        // Create response format
        ResponseFormat responseFormat = ResponseFormat.builder()
                                                      .type(ResponseFormatType.JSON)
                                                      .jsonSchema(jsonSchema)
                                                      .build();

        // Text describing a company
        String text = """
                Tech Innovations Inc. is a software company with several employees.
                John Smith works as a DEVELOPER with email john@tech.com.
                Sarah Johnson is a DESIGNER with email sarah@tech.com.
                Mike Brown serves as a MANAGER with email mike@tech.com.
                Lisa Wang works as an ANALYST with email lisa@tech.com.
                """;

        UserMessage userMessage = UserMessage.from(text);

        // Create chat request
        ChatRequest chatRequest =
                ChatRequest.builder()
                           .responseFormat(responseFormat)
                           .messages(userMessage)
                           .build();

        System.out.println("Extracting company information...");

        // Send request and get response
        ChatResponse chatResponse = chatModel.chat(chatRequest);

        // Extract JSON response
        String jsonResponse = chatResponse.aiMessage().text();
        System.out.println("JSON Response: " + jsonResponse);

        // Parse JSON to Company object
        ObjectMapper mapper = new ObjectMapper();
        try {
            Company company = mapper.readValue(jsonResponse, Company.class);
            System.out.println(company);
            System.out.println("Number of Employees: " + company.employees().size());
            System.out.println("\nEmployees:");
            for (Employee emp : company.employees()) {
                System.out.println(emp);
            }
        } catch (Exception e) {
            System.err.println("Error parsing JSON: " + e.getMessage());
            System.out.println("\nRaw JSON output demonstrates complex nested structure.");
        }
    }
}